﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels.Filters;
    using Utilities;
    using Hims.Shared.Library.Enums;
    using Hims.Api.Models;

    /// <inheritdoc />
    [AllowAnonymous]
    [Route("api/radiology")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class RadiologyController : BaseController
    {
        /// <summary> The radiology service.</summary>
        private readonly IRadiologyService radiologyService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public RadiologyController(IRadiologyService radiologyService, IAuditLogService auditLogServices) { this.radiologyService = radiologyService; this.auditLogServices = auditLogServices; }

        /// <summary>
        /// The fetch Radiology.
        /// </summary>
        /// <param name="model">
        /// The LabOrder filter model.
        /// </param>
        /// <returns>
        /// The list of LabOrders.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of Radiologies.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<RadiologyModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]RadiologyFilterModel model)
        {
            model = (RadiologyFilterModel)EmptyFilter.Handler(model);
            var radiologies = await this.radiologyService.FetchAsync(model);
            return this.Success(radiologies);
        }

        /// <summary>
        /// The add radiology.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - radiology added successfully.
        /// - 409 - radiology already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] RadiologyModel model, [FromHeader] LocationHeader header)
        {
            model = (RadiologyModel)EmptyFilter.Handler(model);
            var response = await this.radiologyService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Radiology Name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.Radiologies,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Name} Radiology has been added successfully.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(" Radiology has been added successfully.");
        }

        /// <summary>
        /// The update radiology.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - radiology updated successfully.
        /// - 409 - radiology already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] RadiologyModel model, [FromHeader] LocationHeader header)
        {
            model = (RadiologyModel)EmptyFilter.Handler(model);
            var response = await this.radiologyService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given  Radiology name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Radiologies,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Name} Radiology has been updated successfully.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(" Radiology has been updated successfully.");
        }

        /// <summary>
        /// The delete radiology.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - radiology deleted successfully.
        /// - 409 - radiology can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] RadiologyModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (RadiologyModel)EmptyFilter.Handler(model);
                var response = await this.radiologyService.DeleteAsync(model.RadiologyId);
                if (response == 0)
                {
                    return this.ServerError();
                }
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Radiologies,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow,
                    LogDescription = $"{model.Name} Radiology has been deleted successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success(" Radiology has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                return this.ServerError(exception.Message);
            }
        }
    }
}